package App::I18N::Command::Gen;
use warnings;
use strict;
use Cwd;
use App::I18N::Config;
use App::I18N::Logger;
use File::Basename;
use File::Path qw(mkpath);
use File::Find::Rule;
use DateTime;
use base qw(App::I18N::Command);

sub options { (
    'podir=s'  => 'podir',
    'locale'  => 'locale',
    'output=s' => 'output_dir',
    ) }


# XXX: provide template option ?

=pod

Generate inline i18n dictionary dictionary:

For Perl:

    # This is auto-generated by App::I18N
    # ------------------------------

    package {{app}}::dict::en;
    our $DICT = {
        "msgid" => "msgstr",
        ...
    };

    package {{app}}::dict::zh_tw;
    our $DICT = {
        "msgid" => "msgstr",
        ....
    };
    ...

For PHP:

    en.php:
    <?php
    $DICT = array( 
        "msgid" => "msgstr"
        ....
    );
    ?>
    ....

For JSON

    en.json:
    {
        "msgid..." : "msgstr"
        ...
    };

    zh_tw.json:
    {
        ....
    };

Static JS

    var dict;
    dict["en"] = { ...  };
    dict["zh_tw"] = { ... };

=cut


sub _warnings {
    my $now = DateTime->now;
    return <<END;
THIS FILE IS AUTO-GENERATED BY APP::I18N, PLEASE DONT MODIFY THIS DIRECTLY.
YOU SHOULD MODIFY PO FILES.
LAST UPDATE: $now
END
}

sub run {
    my ( $self, $type ) = @_;

    my $output_dir = $self->{output_dir} || getcwd();
    my $logger = App::I18N->logger();
    my $podir = $self->{podir};
    $podir = App::I18N->guess_podir( $self ) unless $podir;
    $self->{mo} = 1 if $self->{locale};

    use Encode;
    File::Path::mkpath [ $output_dir ];

    if( $self->{locale} ) {


    }
    else {
        my $warnings = $self->_warnings;
        my @pofiles = File::Find::Rule->file->name( "*.po" )->in( $podir );
        for my $pofile ( @pofiles ) {
            my $extract = Locale::Maketext::Extract->new;
            my ($lang) = ($pofile =~ m{(\w+)\.po$} );   # get en_US or zh_TW ... etc
            my $ext = $type;  # "json , js, pl, pm, php";
            my $outfile = File::Spec->join( $output_dir , "$lang.$ext" );

            $logger->info( "Reading po file: $pofile" );
            $extract->read_po($pofile);

            my $lexicon = $extract->lexicon;
            my %entries = map {   
                # Encode::_utf8_on( $lexicon->{ $_ } );
                my $msgstr = decode_utf8 ( $lexicon->{ $_ } || "" );
                $msgstr
                    ? ( $_ => $msgstr )
                    : () 
                } keys %$lexicon;

            $logger->info( "Writing: $outfile" );
            open FH , ">" , $outfile or die $!;
            # binmode FH,":utf8";
            if( $type eq 'json' ) {
                use JSON::XS;
                print FH encode_json( \%entries );
            }
            elsif ( $type eq 'js' ) {
                use JSON::XS;
                print FH qq|/* $warnings */\n\n|;
                print FH <<END;
var DICT = @{[ encode_json( $lexicon ) ]};
END
            }
            elsif ( $type eq 'pm' ) {
                $warnings =~ s{^}{# }gm;
                use Data::Dumper;
                $Data::Dumper::Terse = 1;
                print FH "package dict::$lang;\n";
                print FH $warnings;
                print FH "\n\n";
                print FH <<END;
our %DICT = @{[ Dumper( $lexicon ) ]};
END
                print FH "1;\n";
            }
            else {
                die "Type: $type is not supported";
            }
            close FH;
        }
    }
}

1;
__END__
=head1 NAME

App::I18N::Command::Gen - Export dictionary to other formats.

=head1 USAGE

    po gen [TYPE] [OPTIONS]

TYPE:

Can be C<json>, C<js>, C<pm>.

=head1 OPTIONS

    --podir=[path]
            Po files directory

    --locale
            Use locale directory structure.

    --output=[path]
            Path for output.

=cut
