#!/usr/bin/env perl
use strict;
use warnings;
use Benchmark qw(cmpthese timethese);
use lib 'blib/lib', 'blib/arch';

use slot qw(slot_counter);

# Package variable for comparison
our $pkg_var = 0;

# Lexical for comparison
my $lexical = 0;

# Hash-based "object"
my $hash = { counter => 0 };

# Array-based "object"
my $array = [0];

# Pure Perl accessor (hash-based OO style)
{
    package PurePerl::Hash;
    sub new { bless { counter => 0 }, shift }
    sub counter {
        my $self = shift;
        $self->{counter} = shift if @_;
        $self->{counter};
    }
}

# Pure Perl accessor (closure/inside-out style)
{
    package PurePerl::Closure;
    my %data;
    sub new { my $x = 0; bless \$x, shift }
    sub counter {
        my $self = shift;
        $data{$$self} = shift if @_;
        $data{$$self};
    }
    sub DESTROY { delete $data{${$_[0]}} }
}

# Pure Perl with lvalue
{
    package PurePerl::Lvalue;
    my %data;
    sub new { my $x = 0; bless \$x, shift }
    sub counter : lvalue {
        my $self = shift;
        $data{$$self};
    }
    sub DESTROY { delete $data{${$_[0]}} }
}

my $pp_hash = PurePerl::Hash->new;
my $pp_closure = PurePerl::Closure->new;
my $pp_lvalue = PurePerl::Lvalue->new;

# Initialize
$pp_closure->counter(0);
$pp_lvalue->counter = 0;

print "=== SETTER BENCHMARK ===\n\n";

cmpthese(-2, {
    'slot'        => sub { slot_counter(1) },
    'slot_lvalue' => sub { slot_counter() = 1 },
    'pp_hash'       => sub { $pp_hash->counter(1) },
    'pp_closure'    => sub { $pp_closure->counter(1) },
    'pp_lvalue'     => sub { $pp_lvalue->counter = 1 },
    'raw_hash'      => sub { $hash->{counter} = 1 },
});

print "\n=== GETTER BENCHMARK ===\n\n";

cmpthese(-2, {
    'slot'      => sub { my $x = slot_counter() },
    'pp_hash'     => sub { my $x = $pp_hash->counter },
    'pp_closure'  => sub { my $x = $pp_closure->counter },
    'pp_lvalue'   => sub { my $x = $pp_lvalue->counter },
    'raw_hash'    => sub { my $x = $hash->{counter} },
});

print "\n=== INCREMENT BENCHMARK ===\n\n";

slot_counter(0);
$pp_hash->counter(0);
$pp_closure->counter(0);
$pp_lvalue->counter = 0;
$hash->{counter} = 0;

cmpthese(-2, {
    'slot_lvalue' => sub { slot_counter()++ },
    'pp_lvalue'     => sub { $pp_lvalue->counter++ },
    'raw_hash'      => sub { $hash->{counter}++ },
});

print "\n=== VS MOO (if available) ===\n\n";

eval {
    require Moo;

    package MooCounter;
    use Moo;
    has counter => (is => 'rw', default => 0);

    package main;
    my $moo = MooCounter->new;

    cmpthese(-2, {
        'slot'        => sub { slot_counter(1) },
        'slot_lvalue' => sub { slot_counter() = 1 },
        'pp_hash'       => sub { $pp_hash->counter(1) },
        'moo'           => sub { $moo->counter(1) },
    });

    1;
} or print "Moo not available\n";
