/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* range.c
*/


#include <string.h>
#include <stdlib.h>

#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "range.h"

static
int RSSize(void *rv)
{
  int size;
  Range *r = (Range*) rv;
  size = sizeof(r) + r->nalo*sizeof(Check);

  return size;
}

vtype rangeRec = {
  "range", NULL, RSSize, NULL, NULL
};


int
VIsNotRange (Value * value)
{
  return !(value
	   && value->value
	   && TypeIndex (value->type) == StringType
	   && strpbrk (value->value, "|:!"));
}


Value *
VInRange (Range * range, Value * value)
{
  int i;
  char *a = NULL;

  if (!VTypeList[TypeIndex (range->type)]->compar)
    {
      parerr = PARCANTRANGE;
      return NULL;
    }

  if (range->type != value->type)
    {
      fprintf (stderr, "INTERNAL TYPE MISMATCH range != value\n");
      exit (2);
    }

  /* Check is the value is an alowable sub-string of
   ** an enumeration
   **/
  if (TypeIndex (range->type) == StringType)
    {

      for (i = 0; i < range->nval; i++)
	{
	  if (!strcmp (range->check[i].v.valu, value->value))
	    return value;

	  if (strstr (range->check[i].v.valu, value->value)
	      == range->check[i].v.valu)
	    {
	      if (a)
		{
		  parerr = PARENUMUNIQ;
		  return NULL;
		}
	      else
		a = range->check[i].v.valu;
	    }
	}
      if (a)
	return VNewValue (value, a, StringType);
      else
	{
	  parerr = PARBADENUM;
	  return NULL;
	}
    }


  for (i = 0; i < range->nval; i++)
    if (VCompar
	(range->type, value->value, range->check[i].v.valu,
	 range->check[i].oper))
      return value;

  parerr = PARLIMITRANGE;
  return NULL;
}


Range *
RangeNew (VType type)
{
  Range *evalue;

  evalue = (Range *) malloc (sizeof (Range));

  evalue->type = type;
  evalue->nval = 0;
  evalue->nalo = 0;

  return evalue;
}


Range *
RangeAdd (Range * range, void *valu, int oper)
{
  if (range->nval >= range->nalo)
    {
      if (range != NULL)
	range = (Range *) realloc (range,
				   sizeof (Range) + (range->nalo +
						     20) * sizeof (Check));
      else
	range =
	  (Range *) malloc (sizeof (Range) +
			    (range->nalo + 20) * sizeof (Check));

      range->nalo += 20;
    }

  memmove (range->check[range->nval].v.valu, valu, VSize (range->type, valu));
  range->check[range->nval++].oper = oper;

  return range;
}


Value *
VRange (ParamFile pfile, Parameter * param, Value * value)
{
  char string[SZ_PFLINE];
  char *tok;
  void *xvalue;
  VType type;
  Range *range = NULL;


  type = param->ptype;

  if (value->type == RangeType)
    return value;

  VDirect (pfile, param, value);

  if (VIsNotRange (value))
    return NULL;

  strcpy (string, value->value);

  range = RangeNew (type);

  for (tok = strtok (string, "|"); tok; tok = strtok (NULL, "|"))
    {
      xvalue = Convert (tok, StringType, NULL, type);

      if (xvalue)
	range = RangeAdd (range, xvalue, PRange_EQ);
      else
	{
	  parerr = PARBADRANGE;
	  return NULL;
	}
    }

  range->str = value->value;
  value->type = RangeType;
  value->value = range;

  return value;
}



void *
Range2String (Range * range, char *string)
{
  if (range && string)
    {
      double mybuf[SZ_PFLINE / sizeof (double) + 1];
      int ii;
      char *str;

      *string = '\0';
      str = (char *) mybuf;
      for (ii = 0; ii < range->nval; ii++)
	{
	  *str = '\0';
	  Convert (range->check[ii].v.valu, range->type, mybuf, StringType);

	  if (str)
	    {
	      if (*string != '\0')
		strcat (string, "|");
	      strcat (string, str);
	    }
	}
      return string;
    }
  else
    return NULL;
}



Value *
PRangeMinMax (Value * value, Value * min, Value * max)
{
  if (value == NULL)
    return NULL;

  if ((min && value->type != min->type) || (max && value->type != max->type))
    {
      fprintf (stderr, "INTERNAL TYPE MISMATCH range != value\n");
      exit (2);
    }
  if (min && max && !VCompar (min->type, min->value, max->value, PRange_LE))
    {
      parerr = PARMAXGTMIN;
      return NULL;
    }
  if (min && !VCompar (value->type, value->value, min->value, PRange_GE))
    {
      parerr = PARLIMITMIN;
      return NULL;
    }

  if (max && !VCompar (value->type, value->value, max->value, PRange_LE))
    {
      parerr = PARLIMITMAX;
      return NULL;
    }

  return value;
}
