use 5.016;
use strict;
use warnings FATAL => 'all';

package Attribute::Universal;

# ABSTRACT: Install attribute handlers directly into UNIVERSAL namespace

use Attribute::Handlers 0.99;

our $VERSION = '0.001';    # VERSION

sub import {
    my $class  = shift;
    my $caller = scalar caller;
    my %cfg    = @_;
    foreach my $name ( keys %cfg ) {
        my $cfg = uc( $cfg{$name} );
        ## no critic
        eval qq{
      sub UNIVERSAL::$name : ATTR($cfg) {
        goto &${caller}::ATTRIBUTE;
      }
    };
        ## use critic
        die "cannot install universal attribute $name in $caller: $@" if $@;
    }
}

1;

__END__

=pod

=head1 NAME

Attribute::Universal - Install attribute handlers directly into UNIVERSAL namespace

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    package Disco;
    
    use Attribute::Universal Loud => 'CODE';
    
    sub ATTRIBUTE {
        my ($package, $symbol, $referent, $attr, $data) = @_;
        # See Attrbute::Handlers
    }
    
    # Attribute is installed global
    
    sub Noise : Loud {
        ...
    }

=head1 DESCRIPTION

According to the example above, this module does just this on import:

    sub UNIVERSAL::Load : ATTR(CODE) {
        goto &Disco::ATTRIBUTE;
    }

Hint: the I<redefine> warning is still enabled.

More than one attribute may be defined at import, with any allowed option:

    use Attribute::Universal RealLoud => 'BEGIN,END', TooLoud => 'ANY,RAWDATA';

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
https://github.com/zurborg/libattribute-universal-perl/issues

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

David Zurborg <zurborg@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by David Zurborg.

This is free software, licensed under:

  The ISC License

=cut
